<?php namespace App\Models;

use CodeIgniter\Model;

class CommonModel extends Model
{

    /**
     * Insert data into table
     */
    public function insertData(string $table, array $data)
    {
        return $this->db->table($table)->insert($data);
    }

    /**
     * Update data in a table with conditions
     */
    public function updateData(string $table, array $conditions, array $data)
    {
        return $this->db->table($table)->where($conditions)->update($data);
    }

    /**
     * Delete data from a table with conditions
     */
    public function deleteData(string $table, array $conditions)
    {
        return $this->db->table($table)->where($conditions)->delete();
    }

    /**
     * Get a single row from a table
     */
    public function getRow(string $table, array $conditions = [])
    {
        return $this->db->table($table)->where($conditions)->get()->getRow();
    }

    /**
     * Get all rows from a table with optional conditions and ordering
     */
    public function getAll(string $table, array $conditions = [], string $orderBy = null)
    {
        $builder = $this->db->table($table)->where($conditions);
        if ($orderBy) {
            [$column, $direction] = explode(' ', $orderBy);
            $builder->orderBy($column, $direction);
        }
        return $builder->get()->getResult();
    }

    /**
     * Get paginated results from a table
     */
    public function getPaginated(string $table, array $conditions = [], int $page = 1, int $limit = 10, string $orderBy = null)
    {
        $offset = ($page - 1) * $limit;

        $builder = $this->db->table($table)->where($conditions);
        $totalRecords = $builder->countAllResults(false);

        if ($orderBy) {
            [$column, $direction] = explode(' ', $orderBy);
            $builder->orderBy($column, $direction);
        }

        $data = $builder->limit($limit, $offset)->get()->getResult();

        return [
            'current_page'      => $page,
            'total_pages'       => ceil($totalRecords / $limit),
            'total_records'     => (string)$totalRecords,
            'records_per_page'  => $limit,
            'data'              => $data
        ];
    }

    /**
     * Run raw query with optional parameters
     */
    public function customQuery(string $sql, array $params = [])
    {
        return $this->db->query($sql, $params)->getResult();
    }

    /**
     * Get reviews with users by astrologer id (already added earlier)
     */
    public function getReviewsByAstrologer($astrologer_id, $page = 1, $limit = 10)
    {
        $offset = ($page - 1) * $limit;

        $totalRecords = $this->db->table('ratings')
            ->where('astrologer_id', $astrologer_id)
            ->countAllResults(false);

        $reviews = $this->db->table('ratings r')
            ->select('r.id, r.rating, r.review, r.created_at, u.name as user_name, u.profile_image')
            ->join('users u', 'u.id = r.user_id')
            ->where('r.astrologer_id', $astrologer_id)
            ->orderBy('r.created_at', 'DESC')
            ->limit($limit, $offset)
            ->get()
            ->getResult();

        foreach ($reviews as &$review) {
            $review->profile_image = $review->profile_image
                ? base_url($review->profile_image)
                : base_url('uploads/users/default.png');
        }

        return [
            'current_page'      => $page,
            'total_pages'       => ceil($totalRecords / $limit),
            'total_records'     => (string) $totalRecords,
            'records_per_page'  => $limit,
            'data'              => $reviews,
        ];
    }
}
