<?php

namespace App\Controllers\api\v1;
use App\Controllers\BaseController;
class UserController extends BaseController
{
    public function profile()
	{
		$user = $this->request->user;

		// Fetch user basic info (excluding old languages field)
		$res = $this->db->table('users')
			->select('id, username, name, mobile, gender, dob, birth_time, exact_birth_time, birth_address, current_address, pincode, wallet_amount, profile_image')
			->where('id', $user->id)
			->get()
			->getRow();

		if (!$res) {
			return apiResponse(false, 'User not found.', null, 404);
		}

		// Handle profile image fallback
		$res->profile_image = !empty($res->profile_image)
			? base_url($res->profile_image)
			: base_url('uploads/defaults/user-default.png');

		// Fetch languages from separate mapping table
		$languageNames = $this->db->table('user_languages ul')
			->select('l.id, l.name')
			->join('languages l', 'l.id = ul.language_id')
			->where('ul.user_id', $user->id)
			->get()
			->getResultArray();

		$res->languages_name = $languageNames;

		return apiResponse(true, 'Profile fetched successfully.', $res);
	}

	
	public function list_languages()
	{
		$languages = $this->db->table('languages')
			->select('id, name')
			->orderBy('name', 'ASC')
			->get()
			->getResult();

		return apiResponse(true, 'Languages fetched successfully.', $languages);
	}
	
	public function list_services()
	{
		$services = $this->db->table('services')
			->select('id, name')
			->orderBy('name', 'ASC')
			->get()
			->getResult();

		return apiResponse(true, 'Services fetched successfully.', $services);
	}

	public function update_profile()
	{
		$data = sanitize(getRequestData());

		// Validation rules
		$rules = [
			'name' => 'required',
			'gender' => 'required|in_list[male,female,other]',
			'dob' => 'required',
			'birth_time' => 'required',
			'exact_birth_time' => 'required|in_list[true,false]',
			'birth_address' => 'required',
			'languages' => 'required',
		];

		if (!$this->validateData($data, $rules)) {
			return apiResponse(false, $this->validator->getErrors(), null, 400);
		}
		if (!is_array($data['languages'])) {
			return apiResponse(false, 'Languages must be arrays.', null, 400);
		}

		$user = $this->request->user;
		$submitData=[
			'name' => $data['name'],
			'gender' => $data['gender'],
			'dob' => $data['dob'],
			'birth_time' => $data['birth_time'],
			'exact_birth_time' => $data['exact_birth_time'],
			'birth_address' => $data['birth_address'],
			'current_address' => $data['current_address'] ?? '',
			'pincode' => $data['pincode'] ?? '',
		];
		$this->db->transStart();

		$this->db->table('users')->where('id', $user->id)->update($submitData);

		// First, delete old mappings
		$this->db->table('user_languages')->where('user_id', $user->id)->delete();

		// Then insert new language mappings
		foreach ($data['languages'] as $langId) {
			$this->db->table('user_languages')->insert([
				'user_id' => $user->id,
				'language_id'   => $langId,
			]);
		}

		$this->db->transComplete();

		if ($this->db->transStatus() === false) {
			return apiResponse(false, 'Failed to update profile.', null, 500);
		}

		return apiResponse(true, 'Profile update successfully.');
		
	}
	
	public function update_fcm_token()
	{
		$data = sanitize(getRequestData());

		// Validation rules
		$rules = [
			'fcm_token' => 'required',
		];

		if (!$this->validateData($data, $rules)) {
			return apiResponse(false, $this->validator->getErrors(), null, 400);
		}

		$fcm_token = $data['fcm_token'];

		// Get authenticated user from JWT (middleware should set this)
		$user = $this->request->user;

		// Update FCM token in DB
		$updated = $this->db->table('users')
			->where('id', $user->id)
			->update(['fcm_token' => $fcm_token]);

		if ($updated) {
			return apiResponse(true, 'FCM token updated successfully.');
		}

		return apiResponse(false, 'Failed to update FCM token.', null, 500);
	}

	public function update_profile_image()
	{
		$user = $this->request->user;
		$image = $this->request->getFile('profile_image');

		// Validate input
		if (!$image || !$image->isValid()) {
			return apiResponse(false, 'No image file uploaded or file is invalid.', null, 400);
		}

		// Check file type
		$validTypes = ['jpg', 'jpeg', 'png'];
		if (!in_array($image->getExtension(), $validTypes)) {
			return apiResponse(false, 'Invalid image format. Only JPG, JPEG, PNG allowed.', null, 400);
		}

		// Get old image path
		$oldImage = $this->db->table('users')->select('profile_image')->where('id', $user->id)->get()->getRow('profile_image');

		// Set new file name
		$newName = 'user_' . $user->id . '_' . time() . '.' . $image->getExtension();
		$uploadPath = FCPATH . 'uploads/users/';

		// Create directory if not exists
		if (!is_dir($uploadPath)) {
			mkdir($uploadPath, 0755, true);
		}

		// Move file
		if (!$image->move($uploadPath, $newName)) {
			return apiResponse(false, 'Failed to upload image.', null, 500);
		}

		// Delete old image if exists and not default
		if (!empty($oldImage)) {
			$oldPath = FCPATH . $oldImage;
			if (file_exists($oldPath) && strpos($oldImage, 'default.png') === false) {
				unlink($oldPath);
			}
		}

		// Update user record
		$this->db->table('users')->where('id', $user->id)->update([
			'profile_image' => 'uploads/users/' . $newName
		]);

		return apiResponse(true, 'Profile image updated successfully.', [
			'image_url' => base_url('uploads/users/' . $newName)
		]);
	}
	
		
	public function profile_setup()
	{
		$data = sanitize(getRequestData());

		// Validation rules
		$rules = [
			'name' => 'required',
			'gender' => 'required|in_list[male,female,other]',
			'dob' => 'required',
			'birth_time' => 'required',
			'exact_birth_time' => 'required|in_list[true,false]',
			'birth_address' => 'required',
			'languages' => 'required',
		];

		if (!$this->validateData($data, $rules)) {
			return apiResponse(false, $this->validator->getErrors(), null, 400);
		}
		
		if (!is_array($data['languages'])) {
			return apiResponse(false, 'Languages must be arrays.', null, 400);
		}

		$user = $this->request->user;

		// Update astrologer basic info
		$submitData = [
			'name' => $data['name'],
			'gender' => $data['gender'],
			'dob' => $data['dob'],
			'birth_time' => $data['birth_time'],
			'exact_birth_time' => $data['exact_birth_time'],
			'birth_address' => $data['birth_address'],
		];

		$this->db->transStart();

		$this->db->table('users')->where('id', $user->id)->update($submitData);

		// First, delete old mappings
		$this->db->table('user_languages')->where('user_id', $user->id)->delete();

		// Then insert new language mappings
		foreach ($data['languages'] as $langId) {
			$this->db->table('user_languages')->insert([
				'user_id' => $user->id,
				'language_id'   => $langId,
			]);
		}

		$this->db->transComplete();

		if ($this->db->transStatus() === false) {
			return apiResponse(false, 'Failed to setup profile.', null, 500);
		}

		return apiResponse(true, 'Profile setup successfully completed.');
	}

	
	public function submit_review()
	{
		$data = sanitize(getRequestData());
		$user = $this->request->user;

		// Validation rules
		$rules = [
			'astrologer_id'  => 'required|is_natural_no_zero',
			'rating'         => 'required|greater_than_equal_to[1]|less_than_equal_to[5]',
			'review'         => 'required|min_length[3]',
		];

		// Optional: review_id for update
		if (!empty($data['review_id'])) {
			$rules['review_id'] = 'is_natural_no_zero';
		}

		if (!$this->validateData($data, $rules)) {
			return apiResponse(false, $this->validator->getErrors(), null, 400);
		}

		$submitData = [
			'astrologer_id' => $data['astrologer_id'],
			'user_id'       => $user->id,
			'rating'        => $data['rating'],
			'review'        => $data['review']
		];

		if (!empty($data['review_id'])) {
			// Check review ownership
			$existing = $this->db->table('ratings')
				->where('id', $data['review_id'])
				->where('user_id', $user->id)
				->get()
				->getRow();

			if (!$existing) {
				return apiResponse(false, 'Review not found or not yours to update.', null, 404);
			}

			$this->db->table('ratings')
				->where('id', $data['review_id'])
				->update($submitData);

			return apiResponse(true, 'Review updated successfully.');
		} else {
			// Insert new review
			$this->db->table('ratings')->insert($submitData);
			return apiResponse(true, 'Review submitted successfully.');
		}
	}
	
	public function delete_review()
	{
		$data = sanitize(getRequestData());
		$user = $this->request->user;

		// Validation: review_id is required
		$rules = [
			'review_id' => 'required|is_natural_no_zero'
		];

		if (!$this->validateData($data, $rules)) {
			return apiResponse(false, $this->validator->getErrors(), null, 400);
		}

		// Check if review exists and belongs to this user
		$review = $this->db->table('ratings')
			->where('id', $data['review_id'])
			->where('user_id', $user->id)
			->get()
			->getRow();

		if (!$review) {
			return apiResponse(false, 'Review not found or not authorized to delete.', null, 404);
		}

		// Delete the review
		$this->db->table('ratings')->where('id', $data['review_id'])->delete();

		return apiResponse(true, 'Review deleted successfully.');
	}

	public function list_reviews()
	{
		$data = sanitize(getRequestData());

		// Validation rules
		$rules = [
			'astrologer_id' => 'required|is_natural_no_zero',
			'page'          => 'permit_empty|is_natural_no_zero',
			'limit'         => 'permit_empty|is_natural_no_zero',
		];

		if (!$this->validateData($data, $rules)) {
			return apiResponse(false, $this->validator->getErrors(), null, 400);
		}

		$astrologer_id = $data['astrologer_id'];
		$page          = !empty($data['page']) ? (int)$data['page'] : 1;
		$limit         = !empty($data['limit']) ? (int)$data['limit'] : 10;

		$res=$this->CommonModel->getReviewsByAstrologer($astrologer_id,$page,$limit);

		return apiResponse(true, 'Reviews fetched successfully.', $res);
	}
	
	public function get_all_astrologers()
	{
		$request = sanitize(getRequestData());
		$user = $this->request->user;

		$page = max(1, (int)($request['page'] ?? 1));
		$limit = (int)($request['limit'] ?? 10);
		$offset = ($page - 1) * $limit;

		$builder = $this->db->table('astrologers a')
			->select('a.id, a.name, a.profile_image, a.experience, a.chat_pricing, a.call_pricing, s.name as specialization_name')
			->join('specializations s', 's.id = a.specialization_id', 'left')
			->where('a.status', 'true')
			->where('a.delete_status', 'false');

		// Filters
		if (!empty($request['search'])) {
			$builder->groupStart()
				->like('a.name', $request['search'])
				->orLike('s.name', $request['search'])
				->groupEnd();
		}

		if (!empty($request['specialization_id'])) {
			$builder->where('a.specialization_id', $request['specialization_id']);
		}

		if (!empty($request['gender'])) {
			$builder->where('a.gender', $request['gender']);
		}

		if (!empty($request['max_chat_price'])) {
			$builder->where('a.chat_pricing <=', $request['max_chat_price']);
		}

		if (!empty($request['max_call_price'])) {
			$builder->where('a.call_pricing <=', $request['max_call_price']);
		}

		if (!empty($request['language_ids']) && is_array($request['language_ids'])) {
			$languageIDs = implode(",", array_map('intval', $request['language_ids']));
			$builder->where("EXISTS (
				SELECT 1 FROM astrologer_languages al
				WHERE al.astrologer_id = a.id AND al.language_id IN ($languageIDs)
			)");
		}

		if (!empty($request['service_ids']) && is_array($request['service_ids'])) {
			$serviceIDs = implode(",", array_map('intval', $request['service_ids']));
			$builder->where("EXISTS (
				SELECT 1 FROM astrologer_services asr
				WHERE asr.astrologer_id = a.id AND asr.service_id IN ($serviceIDs)
			)");
		}

		// Count total
		$total = clone $builder;
		$totalCount = $total->countAllResults(false);

		$builder->groupBy('a.id')->limit($limit, $offset);
		$rows = $builder->get()->getResult();

		$responseData = [];

		foreach ($rows as $row) {
			$profileImage = !empty($row->profile_image)
				? base_url($row->profile_image)
				: base_url('uploads/defaults/user-default.png');

			// Languages
			$langs = $this->db->table('astrologer_languages al')
				->select('l.name')
				->join('languages l', 'l.id = al.language_id')
				->where('al.astrologer_id', $row->id)
				->get()->getResultArray();
			$langNames = array_column($langs, 'name');

			// Rating
			$rating = $this->db->table('ratings')
				->select('AVG(rating) as avg_rating')
				->where('astrologer_id', $row->id)
				->get()->getRow();
			$avgRating = round($rating->avg_rating ?? 0, 1);

			// Favorite
			$isFav = $this->db->table('favorite_astrologers')
				->where('user_id', $user->id)
				->where('astrologer_id', $row->id)
				->countAllResults();

			$responseData[] = [
				'id' => (string) $row->id,
				'name' => $row->name,
				'profile_image' => $profileImage,
				'experience' => $row->experience ?? null,
				'chat_pricing' => $row->chat_pricing ?? "0.00",
				'call_pricing' => $row->call_pricing ?? "0.00",
				'specialization_name' => $row->specialization_name ?? null,
				'languages_name' => $langNames,
				'avg_rating' => $avgRating,
				'is_favorite' => $isFav > 0,
				'experience_display' => ($row->experience ?? 0) . ' yrs+ exp',
			];
		}

		return apiResponse(true, 'Astrologers fetched successfully.', [
			"current_page" => $page,
			"total_pages" => ceil($totalCount / $limit),
			"total_records" => (string)$totalCount,
			"records_per_page" => $limit,
			"data" => $responseData
		]);
	}


	public function get_astrologers_by_service()
	{
		$request = sanitize(getRequestData());
		$user = $this->request->user;

		// --- Validate ---
		$rules = [
			'service_id' => 'required|is_natural_no_zero',
			'page'       => 'permit_empty|is_natural',
			'limit'      => 'permit_empty|is_natural',
		];

		if (!$this->validateData($request, $rules)) {
			return apiResponse(false, $this->validator->getErrors(), null, 400);
		}

		$serviceId = (int) $request['service_id'];
		$page      = isset($request['page']) ? max(1, (int)$request['page']) : 1;
		$limit     = isset($request['limit']) ? (int)$request['limit'] : 10;
		$offset    = ($page - 1) * $limit;

		// --- Build base query ---
		$builder = $this->db->table('astrologers a')
			->select('a.id, a.name, a.profile_image, a.experience, a.chat_pricing, a.call_pricing, s.name as specialization_name')
			->join('astrologer_services asr', 'asr.astrologer_id = a.id')
			->join('specializations s', 's.id = a.specialization_id', 'left')
			->where('asr.service_id', $serviceId)
			->where('a.delete_status', 'false')
			->where('a.status', 'true');

		$totalRecords = $builder->countAllResults(false);

		$astrologers = $builder->limit($limit, $offset)->get()->getResult();

		foreach ($astrologers as &$row) {
			// Profile image fallback
			$row->profile_image = !empty($row->profile_image)
				? base_url($row->profile_image)
				: base_url('uploads/defaults/user-default.png');

			// Language names from astrologer_languages
			$languageRows = $this->db->table('astrologer_languages al')
				->select('l.name')
				->join('languages l', 'l.id = al.language_id')
				->where('al.astrologer_id', $row->id)
				->get()->getResultArray();

			$row->languages_name = array_column($languageRows, 'name');

			// Avg rating
			$ratingData = $this->db->table('ratings')
				->select('AVG(rating) as avg_rating')
				->where('astrologer_id', $row->id)
				->get()->getRow();
			$row->avg_rating = round($ratingData->avg_rating ?? 0, 1);

			// Favorite flag
			$row->is_favorite = $this->db->table('favorite_astrologers')
				->where('user_id', $user->id)
				->where('astrologer_id', $row->id)
				->countAllResults() > 0;

			$row->experience_display = $row->experience . ' yrs+ exp';
		}

		return apiResponse(true, 'Astrologers by service fetched.', [
			'current_page'      => $page,
			'total_pages'       => ceil($totalRecords / $limit),
			'total_records'     => $totalRecords,
			'records_per_page'  => $limit,
			'data'              => $astrologers,
		]);
	}

	public function popular_astrologers()
	{
		$data = sanitize(getRequestData());
		$user = $this->request->user;

		$page = max(1, (int)($data['page'] ?? 1));
		$limit = 10;
		$offset = ($page - 1) * $limit;

		// Total count
		$total = $this->db->table('astrologers')
			->where('delete_status', 'false')
			->where('status', 'true')
			->countAllResults();

		// Main query
		$astrologers = $this->db->table('astrologers a')
			->select("a.id, a.name, a.experience, a.chat_pricing, a.call_pricing, a.profile_image,
					  s.name as specialization_name,
					  IFNULL(ROUND(AVG(r.rating), 1), 0) as avg_rating,
					  COUNT(r.id) as total_reviews")
			->join('specializations s', 's.id = a.specialization_id', 'left')
			->join('ratings r', 'r.astrologer_id = a.id', 'left')
			->where('a.delete_status', 'false')
			->where('a.status', 'true')
			->groupBy('a.id')
			->orderBy('avg_rating', 'DESC')
			->orderBy('total_reviews', 'DESC')
			->limit($limit, $offset)
			->get()->getResult();

		$result = [];

		foreach ($astrologers as $astrologer) {
			// Profile image fallback
			$profileImage = !empty($astrologer->profile_image)
				? base_url($astrologer->profile_image)
				: base_url('uploads/defaults/user-default.png');

			// Languages
			$langQuery = $this->db->table('astrologer_languages al')
				->select('l.name')
				->join('languages l', 'l.id = al.language_id')
				->where('al.astrologer_id', $astrologer->id)
				->get()->getResultArray();
			$languageNames = array_column($langQuery, 'name');

			// Favorite check
			$isFav = $this->db->table('favorite_astrologers')
				->where('user_id', $user->id)
				->where('astrologer_id', $astrologer->id)
				->countAllResults();

			$result[] = [
				'id' => (string) $astrologer->id,
				'name' => $astrologer->name,
				'profile_image' => $profileImage,
				'experience' => $astrologer->experience,
				'chat_pricing' => $astrologer->chat_pricing ?? "0.00",
				'call_pricing' => $astrologer->call_pricing ?? "0.00",
				'specialization_name' => $astrologer->specialization_name ?? null,
				'languages_name' => $languageNames,
				'avg_rating' => (float) $astrologer->avg_rating,
				'is_favorite' => $isFav > 0,
				'experience_display' => ($astrologer->experience ?? 0) . ' yrs+ exp',
			];
		}

		$response = [
			'current_page' => $page,
			'total_pages' => ceil($total / $limit),
			'total_records' => (string) $total,
			'records_per_page' => $limit,
			'data' => $result
		];

		return apiResponse(true, 'Popular astrologers fetched successfully.', $response);
	}



	public function astrologer_details()
	{
		$data = sanitize(getRequestData());

		// Validation rules
		$rules = [
			'astrologer_id' => 'required|is_natural_no_zero',
		];

		if (!$this->validateData($data, $rules)) {
			return apiResponse(false, $this->validator->getErrors(), null, 400);
		}

		$astrologerId = $data['astrologer_id'];
		$user = $this->request->user;

		// 1. Fetch astrologer with specialization
		$astrologer = $this->db->table('astrologers a')
			->select('a.id, a.name, a.gender, a.experience, a.about, a.profile_image, a.specialization_id, a.chat_pricing, a.call_pricing, s.name as specialization_name')
			->join('specializations s', 's.id = a.specialization_id', 'left')
			->where('a.id', $astrologerId)
			->where('a.delete_status', 'false')
			->where('a.status', 'true')
			->get()
			->getRow();

		if (!$astrologer) {
			return apiResponse(false, 'Astrologer not found.', null, 404);
		}

		// 2. Format profile image
		$astrologer->profile_image = !empty($astrologer->profile_image)
			? base_url($astrologer->profile_image)
			: base_url('uploads/defaults/user-default.png');

		// 3. Languages from pivot table
		$languageRows = $this->db->table('astrologer_languages al')
			->select('l.name')
			->join('languages l', 'l.id = al.language_id')
			->where('al.astrologer_id', $astrologerId)
			->get()
			->getResultArray();
		$astrologer->language_names = array_column($languageRows, 'name');

		// 4. Services Offered from pivot table
		$serviceRows = $this->db->table('astrologer_services asr')
			->select('s.id,s.name,s.icon')
			->join('services s', 's.id = asr.service_id')
			->where('asr.astrologer_id', $astrologerId)
			->get()
			->getResultArray();
		$astrologer->services = $serviceRows;

		// 5. Recent Reviews (limit 3)
		$reviews = $this->db->table('ratings r')
			->select('r.rating, r.review, u.name, u.profile_image')
			->join('users u', 'u.id = r.user_id')
			->where('r.astrologer_id', $astrologerId)
			->orderBy('r.created_at', 'DESC')
			->limit(3)
			->get()
			->getResult();

		foreach ($reviews as &$review) {
			$review->profile_image = !empty($review->profile_image)
				? base_url($review->profile_image)
				: base_url('uploads/defaults/user-default.png');
		}

		$astrologer->recent_reviews = $reviews;


		// 6. Rating and total reviews
		$ratingData = $this->db->table('ratings')
			->select('AVG(rating) as avg_rating, COUNT(*) as total_reviews')
			->where('astrologer_id', $astrologerId)
			->get()
			->getRow();
		$astrologer->avg_rating = round($ratingData->avg_rating ?? 0, 1);
		$astrologer->total_reviews = (int) $ratingData->total_reviews;

		// 7. Check if favorited
		$isFav = $this->db->table('favorite_astrologers')
			->where('user_id', $user->id)
			->where('astrologer_id', $astrologerId)
			->countAllResults();
		$astrologer->is_favorite = $isFav > 0;


		return apiResponse(true, 'Astrologer details fetched successfully.', $astrologer);
	}


	public function toggle_favorite()
	{
		$data = sanitize(getRequestData());
		$user = $this->request->user;

		if (empty($data['astrologer_id'])) {
			return apiResponse(false, 'Astrologer ID is required.', null, 400);
		}

		$astrologerId = intval($data['astrologer_id']);

		// Check if the astrologer exists
		$exists = $this->db->table('astrologers')->where('id', $astrologerId)->countAllResults();
		if ($exists === 0) {
			return apiResponse(false, 'Astrologer not found.', null, 404);
		}

		// Check if already favorited
		$favTable = $this->db->table('favorite_astrologers');
		$exists = $favTable
			->where('user_id', $user->id)
			->where('astrologer_id', $astrologerId)
			->get()
			->getRow();

		if ($exists) {
			// Remove favorite
			$favTable
				->where('user_id', $user->id)
				->where('astrologer_id', $astrologerId)
				->delete();

			return apiResponse(true, 'Removed from favorites.');
		} else {
			// Add to favorite
			$favTable->insert([
				'user_id'       => $user->id,
				'astrologer_id' => $astrologerId
			]);

			return apiResponse(true, 'Added to favorites.');
		}
	}
	
	public function favorite_list()
	{
		$user = $this->request->user;

		$favorites = $this->db->table('favorite_astrologers fa')
			->select('a.id, a.name, a.profile_image, a.specialization_id, s.name as specialization')
			->join('astrologers a', 'a.id = fa.astrologer_id')
			->join('specializations s', 's.id = a.specialization_id', 'left')
			->where('fa.user_id', $user->id)
			->get()
			->getResult();

		// Format profile image
		foreach ($favorites as &$f) {
			$f->profile_image = !empty($f->profile_image) 
				? base_url($f->profile_image) 
				: base_url('uploads/defaults/user-default.png');
		}

		return apiResponse(true, 'Favorite astrologers fetched.', $favorites);
	}

public function available_slots()
{
    $data = sanitize(getRequestData());

    $rules = [
        'astrologer_id'  => 'required|is_natural_no_zero',
        'date'           => 'required|valid_date[Y-m-d]',
        'slot_duration'  => 'required|is_natural_no_zero' // e.g. 5, 10, 15 mins
    ];

    if (!$this->validateData($data, $rules)) {
        return apiResponse(false, $this->validator->getErrors(), null, 400);
    }

    $slotDuration = (int) $data['slot_duration'];
    if ($slotDuration < 5 || $slotDuration > 60) {
        return apiResponse(false, 'Slot duration must be between 5 and 60 minutes.', null, 422);
    }

    $dayOfWeek = strtolower(date('l', strtotime($data['date'])));

    $avail = $this->db->table('astrologer_availability')
        ->where('astrologer_id', $data['astrologer_id'])
        ->where('day_of_week', $dayOfWeek)
        ->where('is_available', true)
        ->get()->getRow();

    if (!$avail) {
        return apiResponse(true, 'No slots available for the selected day.', []);
    }

    // Get already booked slots for that date
    $booked = $this->db->table('bookings')
        ->select('time_slot')
        ->where('astrologer_id', $data['astrologer_id'])
        ->where('booking_date', $data['date'])
        ->whereIn('status', ['pending', 'confirmed'])
        ->get()->getResultArray();

    $bookedSlots = array_column($booked, 'time_slot'); // ["14:00 - 14:10", "15:00 - 15:15"]
    
    // Generate available slots (excluding overlapping with booked)
    $availableSlots = $this->generateTimeSlots($avail->start_time, $avail->end_time, $slotDuration, $bookedSlots);

    return apiResponse(true, 'Available slots fetched.', $availableSlots);
}

private function generateTimeSlots($startTime, $endTime, $interval, $bookedSlots = [])
{
    $slots = [];
    $start = strtotime($startTime);
    $end = strtotime($endTime);

    // Convert bookedSlots to ranges
    $bookedRanges = [];
    foreach ($bookedSlots as $slot) {
        if (strpos($slot, ' - ') !== false) {
            list($from, $to) = explode(' - ', $slot);
            $bookedRanges[] = [
                'start' => strtotime($from),
                'end'   => strtotime($to)
            ];
        }
    }

    // Loop through and check overlaps
    while ($start + ($interval * 60) <= $end) {
        $slotStart = $start;
        $slotEnd = $start + ($interval * 60);
        $overlap = false;

        foreach ($bookedRanges as $booked) {
            if ($slotStart < $booked['end'] && $slotEnd > $booked['start']) {
                $overlap = true;
                break;
            }
        }

        if (!$overlap) {
            $slots[] = date('H:i', $slotStart) . ' - ' . date('H:i', $slotEnd);
        }

        $start += $interval * 60;
    }

    return $slots;
}


public function book_service()
{
	$data = sanitize(getRequestData());
	$user = $this->request->user;

	$rules = [
		'astrologer_id'  => 'required|is_natural_no_zero',
		'service_id'     => 'required|is_natural_no_zero',
		'booking_date'   => 'required|valid_date[Y-m-d]',
		'time_slot'      => 'required',
		'slot_duration'  => 'required|is_natural_no_zero|greater_than_equal_to[5]|less_than_equal_to[60]',
	];

	if (!$this->validateData($data, $rules)) {
		return apiResponse(false, $this->validator->getErrors(), null, 400);
	}

	// Validate time_slot format: "HH:MM - HH:MM"
	if (!preg_match('/^\d{2}:\d{2} - \d{2}:\d{2}$/', $data['time_slot'])) {
		return apiResponse(false, 'Invalid time slot format. Use "HH:MM - HH:MM".', null, 422);
	}
	list($slotStart, $slotEnd) = explode(' - ', $data['time_slot']);
	if (strtotime($slotStart) >= strtotime($slotEnd)) {
		return apiResponse(false, 'Start time must be before end time.', null, 422);
	}

	// Check astrologer status
	$astro = $this->db->table('astrologers')->where('id', $data['astrologer_id'])->get()->getRow();
	if (!$astro || $astro->delete_status === 'true' || $astro->status !== 'true') {
		return apiResponse(false, 'Astrologer not available.', null, 404);
	}

	

	// Use per_minute_rate from the service
	$perMinuteRate = (float) $astro->call_pricing;
	$amount = $perMinuteRate * (int)$data['slot_duration'];

	$currentBalance = $this->getWalletBalance($user->id);
	if ($currentBalance < $amount) {
		return apiResponse(false, 'Insufficient balance');
	}

	// Check for time overlaps
	$existingBookings = $this->db->table('bookings')
		->select('time_slot')
		->where('astrologer_id', $data['astrologer_id'])
		->where('booking_date', $data['booking_date'])
		->whereIn('status', ['pending', 'confirmed'])
		->get()->getResult();

	foreach ($existingBookings as $booking) {
		if (strpos($booking->time_slot, ' - ') !== false) {
			list($bookedStart, $bookedEnd) = explode(' - ', $booking->time_slot);
			$bookedStart = strtotime($bookedStart);
			$bookedEnd = strtotime($bookedEnd);
			$start = strtotime($slotStart);
			$end = strtotime($slotEnd);

			// Check overlap
			if ($start < $bookedEnd && $end > $bookedStart) {
				return apiResponse(false, 'Selected time slot overlaps with existing booking.', null, 409);
			}
		}
	}

	// Store booking
	$bookingData = [
		'user_id'       => $user->id,
		'astrologer_id' => $data['astrologer_id'],
		'service_id'    => $data['service_id'],
		'booking_date'  => $data['booking_date'],
		'time_slot'     => $data['time_slot'],
		'slot_duration' => $data['slot_duration'],
		'amount'        => $amount,
		'status'        => 'pending',
		'created_at'    => $this->data['datetime']
	];

	$inserted = $this->db->table('bookings')->insert($bookingData);

	if ($inserted) {
		$booking_id = $this->db->insertID();
		$booking_id=generateBookingId($booking_id);
		$newBalance = $currentBalance - $amount;
		$this->db->table('wallet_transactions')->insert([
			'user_id' => $user->id,
			'payment_id' => $booking_id,
			'transaction_type' => 'debit',
			'amount' => $amount,
			'opening_balance' => $currentBalance,
			'closing_balance' => $newBalance,
			'description' => 'Service booking for astrologer Booking ID: ' . $booking_id,
			'created_at'=>$this->data['datetime']
		]);

		// Update wallet in users
		$this->db->table('users')->where('id', $user->id)->update(['wallet_amount' => $newBalance]);

		return apiResponse(true, 'Booking confirmed successfully.', [
			'booking_id' => $booking_id,
			'amount'     => $amount,
			'slot'       => $data['time_slot']
		]);
	}

	return apiResponse(false, 'Failed to create booking.', null, 500);
}

public function cancel_booking()
{
	$data = sanitize(getRequestData());
	$user = $this->request->user;
	$rules = [
		'booking_id'   => 'required'
	];

	if (!$this->validateData($data, $rules)) {
		return apiResponse(false, $this->validator->getErrors(), null, 400);
	}
	$booking = $this->db->table('bookings')->where([
		'id'      => $data['booking_id'],
		'user_id' => $user->id,
	])->get()->getRow();

	if (!$booking || $booking->status === 'cancelled') {
		return apiResponse(false, 'Booking not found or already cancelled.');
	}

	$this->db->transStart();

	// 1. Update booking status
	$this->db->table('bookings')->where('id', $booking->id)->update([
		'status'   => 'cancelled',
		'cancelled_by'   => 'user',
		'updated_at' => $this->data['datetime']
	]);

	// 2. Refund if not already refunded
	if ((int)$booking->refunded === 0) {
		// Get user's current wallet balance
		$userRow = $this->db->table('users')->where('id', $user->id)->get()->getRow();
		$opening_balance = (float)$userRow->wallet_amount;
		$refund_amount   = (float)$booking->amount;
		$closing_balance = $opening_balance + $refund_amount;

		// Insert into wallet_transactions
		$this->db->table('wallet_transactions')->insert([
			'user_id'          => $user->id,
			'payment_id'       => $booking->booking_id, // no payment_id for refund
			'transaction_type' => 'credit',
			'amount'           => $refund_amount,
			'opening_balance'  => $opening_balance,
			'closing_balance'  => $closing_balance,
			'description'      => 'Refund for cancelled booking ID ' . $booking->booking_id,
			'created_at'       => $this->data['datetime']
		]);

		// Update user's wallet balance
		$this->db->table('users')->where('id', $user->id)->update([
			'wallet_amount' => $closing_balance
		]);

		// Update refunded flag
		$this->db->table('bookings')->where('id', $booking->id)->update([
			'refunded' => 1
		]);
	}

	$this->db->transComplete();

	if (!$this->db->transStatus()) {
		return apiResponse(false, 'Cancellation failed due to internal error.', null, 500);
	}

	return apiResponse(true, 'Booking cancelled and amount refunded successfully.');
}



public function create_wallet_order()
{
	$data = sanitize(getRequestData());
	$user = $this->request->user;
	$rules = [
		'amount' => 'required|numeric|greater_than_equal_to[10]'
	];

	if (!$this->validateData($data, $rules)) {
		return apiResponse(false, $this->validator->getErrors(), null, 400);
	}
	$amount = (float)$data['amount'];

	$order = $this->razorpay->order->create([
		'receipt' => 'wallet_' . time(),
		'amount'  => $amount * 100,
		'currency' => 'INR'
	]);

	// Save in DB
	$this->db->table('payment_transactions')->insert([
		'user_id' => $user->id,
		'amount' => $amount,
		'order_id' => $order['id'],
		'status' => 'created'
	]);

	return apiResponse(true, 'Order created', [
		'order_id' => $order['id']
	]);
}

public function verify_wallet_payment()
{
	$data = sanitize(getRequestData());
	$user = $this->request->user;

	$rules = [
		'order_id'    => 'required',
		'payment_id'  => 'required',
		'signature'   => 'required',
	];

	if (!$this->validateData($data, $rules)) {
		return apiResponse(false, $this->validator->getErrors(), null, 400);
	}

	// Fetch payment
	$payment = $this->db->table('payment_transactions')->where([
		'user_id' => $user->id,
		'order_id' => $data['order_id']
	])->get()->getRow();

	if (!$payment || $payment->status == 'success') {
		return apiResponse(false, 'Invalid or already verified transaction');
	}

	// Razorpay Signature verification (optional if using client confirmation)
	$expected_signature = hash_hmac('sha256', $data['order_id'] . '|' . $data['payment_id'], env('razorpay_key_secret'));
	if ($expected_signature !== $data['signature']) {
		return apiResponse(false, 'Signature verification failed');
	}

	// Transaction verified
	$this->db->transStart();

	// Update payment
	$this->db->table('payment_transactions')->where('id', $payment->id)->update([
		'payment_id' => $data['payment_id'],
		'signature' => $data['signature'],
		'payment_response' => json_encode($data),
		'status' => 'success'
	]);

	// Get current balance
	$currentBalance = $this->getWalletBalance($user->id);
	$newBalance = $currentBalance + $payment->amount;

	// Insert wallet credit
	$this->db->table('wallet_transactions')->insert([
		'user_id' => $user->id,
		'payment_id' => $payment->id,
		'transaction_type' => 'credit',
		'amount' => $payment->amount,
		'opening_balance' => $currentBalance,
		'closing_balance' => $newBalance,
		'description' => 'Wallet recharge via Razorpay'
	]);

	// Update user wallet
	$this->db->table('users')->where('id', $user->id)->update(['wallet_amount' => $newBalance]);

	$this->db->transComplete();

	if (!$this->db->transStatus()) {
		return apiResponse(false, 'Failed to update wallet', null, 500);
	}

	return apiResponse(true, 'Wallet recharged successfully');
}

private function getWalletBalance($userId)
{
	return (float)($this->db->table('users')->select('wallet_amount')->where('id', $userId)->get()->getRow()->wallet_amount ?? 0);
}

public function get_booked_sessions()
{
    $user = $this->request->user; // or get from token

    $sessions = $this->db->table('bookings b')
        ->select("
            b.id,
            b.booking_id,
            b.booking_date,
            b.time_slot,
            b.slot_duration,
            b.amount,
            a.name as astrologer_name,
            a.id as astrologer_id,
            a.profile_image,
            a.experience,
            a.call_pricing,
            a.chat_pricing,
            s.name as specialization_name,
            IFNULL(ROUND(AVG(r.rating), 1), 0) as avg_rating,
            COUNT(r.id) as total_reviews
        ")
        ->join('astrologers a', 'a.id = b.astrologer_id', 'left')
        ->join('specializations s', 's.id = a.specialization_id', 'left')
        ->join('ratings r', 'r.astrologer_id = a.id', 'left')
        ->where('b.user_id', $user->id)
        ->where('b.status', 'confirmed')
        ->groupBy('b.id')
        ->orderBy('b.booking_date', 'DESC')
        ->orderBy('b.id', 'DESC')
        ->get()
        ->getResult();

    // Format response
    foreach ($sessions as &$s) {
		$s->booking_date=date('d M Y, h:i A',strtotime($s->booking_date.' '.explode('-',$s->time_slot)[0]));
		$s->profile_image = !empty($s->profile_image)
				? base_url($s->profile_image)
				: base_url('uploads/defaults/user-default.png');
		$s->experience_display = ($s->experience ?? 0) . ' yrs+ exp';

		$langQuery = $this->db->table('astrologer_languages al')
				->select('l.name')
				->join('languages l', 'l.id = al.language_id')
				->where('al.astrologer_id', $s->astrologer_id)
				->get()->getResultArray();
			$s->languages_name = array_column($langQuery, 'name');
    }

    return apiResponse(true, 'Booked sessions fetched successfully', $sessions);
}

public function astrologer_recently_talk()
{
    $user = $this->request->user; // or get from token

    $sessions = $this->db->table('bookings b')
        ->select("
            b.id,
            b.booking_id,
            b.booking_date,
            b.time_slot,
            a.name as astrologer_name,
            a.id as astrologer_id,
            a.profile_image,
            a.experience,
            a.call_pricing,
            a.chat_pricing,
            s.name as specialization_name,
            IFNULL(ROUND(AVG(r.rating), 1), 0) as avg_rating,
            COUNT(r.id) as total_reviews
        ")
        ->join('astrologers a', 'a.id = b.astrologer_id', 'left')
        ->join('specializations s', 's.id = a.specialization_id', 'left')
        ->join('ratings r', 'r.astrologer_id = a.id', 'left')
        ->where('b.user_id', $user->id)
        ->where('b.status', 'completed')
        ->groupBy('b.id')
        ->orderBy('b.booking_date', 'DESC')
        ->orderBy('b.id', 'DESC')
        ->get()
        ->getResult();

    // Format response
    foreach ($sessions as &$s) {
		$s->last_talked='Last talk '.timeAgo($s->booking_date.' '.explode('-',$s->time_slot)[0]);
		$s->profile_image = !empty($s->profile_image)
				? base_url($s->profile_image)
				: base_url('uploads/defaults/user-default.png');
		$s->experience_display = ($s->experience ?? 0) . ' yrs+ exp';

		$langQuery = $this->db->table('astrologer_languages al')
				->select('l.name')
				->join('languages l', 'l.id = al.language_id')
				->where('al.astrologer_id', $s->astrologer_id)
				->get()->getResultArray();
			$s->languages_name = array_column($langQuery, 'name');
    }

    return apiResponse(true, 'Booked sessions fetched successfully', $sessions);
}


public function all_booked_sessions()
{
    $user = $this->request->user; // or get from token

    $sessions = $this->db->table('bookings b')
        ->select("
            b.id,
            b.booking_id,
            b.booking_date,
            b.time_slot,
            b.slot_duration,
            b.amount,
            b.status,
            a.name as astrologer_name,
            a.id as astrologer_id,
            a.profile_image,
            a.experience,
            a.call_pricing,
            a.chat_pricing,
            s.name as specialization_name,
            IFNULL(ROUND(AVG(r.rating), 1), 0) as avg_rating,
            COUNT(r.id) as total_reviews
        ")
        ->join('astrologers a', 'a.id = b.astrologer_id', 'left')
        ->join('specializations s', 's.id = a.specialization_id', 'left')
        ->join('ratings r', 'r.astrologer_id = a.id', 'left')
        ->where('b.user_id', $user->id)
        ->whereIn('b.status', ['confirmed', 'pending'])
        ->groupBy('b.id')
        ->orderBy('b.booking_date', 'DESC')
        ->orderBy('b.id', 'DESC')
        ->get()
        ->getResult();

    // Format response
    foreach ($sessions as &$s) {
		$s->booking_date=date('d M Y, h:i A',strtotime($s->booking_date.' '.explode('-',$s->time_slot)[0]));
		$s->profile_image = !empty($s->profile_image)
				? base_url($s->profile_image)
				: base_url('uploads/defaults/user-default.png');
		$s->experience_display = ($s->experience ?? 0) . ' yrs+ exp';

		$langQuery = $this->db->table('astrologer_languages al')
				->select('l.name')
				->join('languages l', 'l.id = al.language_id')
				->where('al.astrologer_id', $s->astrologer_id)
				->get()->getResultArray();
			$s->languages_name = array_column($langQuery, 'name');
    }

    return apiResponse(true, 'Booked sessions fetched successfully', $sessions);
}


public function booking_history()
{
    $user = $this->request->user; // Get user from token or request

    // Get pagination params
    $page  = (int) ($this->request->getVar('page') ?? 1);
    $limit = (int) ($this->request->getVar('limit') ?? 10);
    $offset = ($page - 1) * $limit;

    // Optional date range filter
    $startDate = $this->request->getVar('start_date'); // format: YYYY-MM-DD
    $endDate   = $this->request->getVar('end_date');   // format: YYYY-MM-DD

    // Base query for count
    $countBuilder = $this->db->table('bookings b')
        ->select('COUNT(DISTINCT b.id) as total')
        ->where('b.user_id', $user->id);

    if ($startDate && $endDate) {
        $countBuilder->where("DATE(b.booking_date) >=", $startDate)
                     ->where("DATE(b.booking_date) <=", $endDate);
    }

    $total = $countBuilder->get()->getRow()->total;

    // Main query for fetching paginated results
    $sessions = $this->db->table('bookings b')
        ->select("
            b.id,
            b.booking_id,
            b.booking_date,
            b.time_slot,
            b.slot_duration,
            b.amount,
            b.status,
            a.name as astrologer_name,
            a.id as astrologer_id,
            a.profile_image,
            a.experience,
            a.call_pricing,
            a.chat_pricing,
            s.name as specialization_name,
            IFNULL(ROUND(AVG(r.rating), 1), 0) as avg_rating,
            COUNT(r.id) as total_reviews
        ")
        ->join('astrologers a', 'a.id = b.astrologer_id', 'left')
        ->join('specializations s', 's.id = a.specialization_id', 'left')
        ->join('ratings r', 'r.astrologer_id = a.id', 'left')
        ->where('b.user_id', $user->id);

    if ($startDate && $endDate) {
        $sessions->where("DATE(b.booking_date) >=", $startDate)
                 ->where("DATE(b.booking_date) <=", $endDate);
    }

    $sessions = $sessions->groupBy('b.id')
        ->orderBy('b.booking_date', 'DESC')
        ->orderBy('b.id', 'DESC')
        ->limit($limit, $offset)
        ->get()
        ->getResult();

    // Format data
    foreach ($sessions as &$s) {
        $s->booking_date = date('d M Y, h:i A', strtotime($s->booking_date . ' ' . explode('-', $s->time_slot)[0]));
        $s->profile_image = !empty($s->profile_image)
            ? base_url($s->profile_image)
            : base_url('uploads/defaults/user-default.png');
        $s->experience_display = ($s->experience ?? 0) . ' yrs+ exp';

        $langQuery = $this->db->table('astrologer_languages al')
            ->select('l.name')
            ->join('languages l', 'l.id = al.language_id')
            ->where('al.astrologer_id', $s->astrologer_id)
            ->get()->getResultArray();

        $s->languages_name = array_column($langQuery, 'name');
    }

    // Final formatted response
    $response = [
        'current_page'      => (string)$page,
        'total_pages'       => (string)ceil($total / $limit),
        'total_records'     => (string) $total,
        'records_per_page'  => (string)$limit,
        'data'              => $sessions
    ];

    return apiResponse(true, 'Booking history fetched successfully', $response);
}
public function wallet_history()
{
    $user = $this->request->user; // Authenticated user

    $page  = (int) ($this->request->getVar('page') ?? 1);
    $limit = (int) ($this->request->getVar('limit') ?? 10);
    $offset = ($page - 1) * $limit;

    $startDate = $this->request->getVar('start_date'); // format: YYYY-MM-DD
    $endDate   = $this->request->getVar('end_date');   // format: YYYY-MM-DD
    $transaction_type = $this->request->getVar('transaction_type'); // 'credit' or 'debit'
    $search = $this->request->getVar('search'); // keyword search

    // Count query
    $countBuilder = $this->db->table('wallet_transactions')
        ->select('COUNT(*) as total')
        ->where('user_id', $user->id);

    if ($startDate && $endDate) {
        $countBuilder->where("DATE(created_at) >=", $startDate)
                     ->where("DATE(created_at) <=", $endDate);
    }

    if ($transaction_type) {
        $countBuilder->where('transaction_type', $transaction_type);
    }

    if ($search) {
        $countBuilder->groupStart()
            ->like('description', $search)
            ->orLike('payment_id', $search)
            ->groupEnd();
    }

    $total = $countBuilder->get()->getRow()->total;

    // Main data query
    $wallets = $this->db->table('wallet_transactions')
        ->select('id, payment_id, transaction_type, amount, opening_balance, closing_balance, description, created_at')
        ->where('user_id', $user->id);

    if ($startDate && $endDate) {
        $wallets->where("DATE(created_at) >=", $startDate)
                ->where("DATE(created_at) <=", $endDate);
    }

    if ($transaction_type) {
        $wallets->where('transaction_type', $transaction_type);
    }

    if ($search) {
        $wallets->groupStart()
            ->like('description', $search)
            ->orLike('payment_id', $search)
            ->groupEnd();
    }

    $wallets = $wallets->orderBy('created_at', 'DESC')
        ->limit($limit, $offset)
        ->get()
        ->getResult();

    // Format date
    foreach ($wallets as &$w) {
        $w->created_at = date('d M Y, h:i A', strtotime($w->created_at));
    }

    $response = [
        'current_page'      => (string)$page,
        'total_pages'       => (string)ceil($total / $limit),
        'total_records'     => (string)$total,
        'records_per_page'  => (string)$limit,
        'data'              => $wallets,
    ];

    return apiResponse(true, 'Wallet history fetched successfully', $response);
}



	public function logout()
	{
		$user = $this->request->user;

		// Delete all access/refresh tokens
		$this->db->table('user_tokens')
			->where('user_id', $user->id)
			->where('user_type', $user->user_type)
			->delete();

		// Blank out FCM token in users table
		$this->db->table('users')
			->where('id', $user->id)
			->update(['fcm_token' => null]);

		return apiResponse(true, 'Logged out successfully.');
	}

	public function delete_account()
	{
		$user = $this->request->user;

		// Delete all access/refresh tokens
		$this->db->table('user_tokens')
			->where('user_id', $user->id)
			->where('user_type', $user->user_type)
			->delete();

		// Blank out FCM token in users table
		$this->db->table('users')
			->where('id', $user->id)
			->update(['fcm_token' => null,'delete_status'=>'true']);

		return apiResponse(true, 'Account deleted successfully.');
	}

    
}
