<?php

namespace App\Controllers\api\v1;
use App\Controllers\BaseController;
class AstrologerController extends BaseController
{
    public function profile()
{
	$user = $this->request->user;

	// Fetch astrologer with specialization
	$res = $this->db->table('astrologers a')
		->select('a.id, a.username, a.name, a.mobile, a.specialization_id, s.name as specialization_name, a.experience, a.about, a.address, a.pincode, a.wallet_amount, a.profile_image, a.chat_pricing, a.call_pricing')
		->join('specializations s', 's.id = a.specialization_id', 'left')
		->where('a.id', $user->id)
		->get()
		->getRow();

	if (!$res) {
		return apiResponse(false, 'Astrologer not found.', null, 404);
	}

	// Handle profile image
	$res->profile_image = !empty($res->profile_image)
		? base_url($res->profile_image)
		: base_url('uploads/defaults/user-default.png');

	// ✅ Fetch languages
	$languageNames = $this->db->table('astrologer_languages al')
		->select('l.id,l.name')
		->join('languages l', 'l.id = al.language_id')
		->where('al.astrologer_id', $user->id)
		->get()
		->getResultArray();

	$res->languages_name = $languageNames;

	// ✅ Fetch services
	$serviceNames = $this->db->table('astrologer_services asv')
		->select('sv.id,sv.name,sv.icon')
		->join('services sv', 'sv.id = asv.service_id')
		->where('asv.astrologer_id', $user->id)
		->get()
		->getResultArray();

	$res->services_name =$serviceNames;
	$res->upcoming_booking =$this->getUpcomingBookings($user->id,3);
	$res->recent_booking =$this->getRecentBooking($user->id,3);
	$res->recent_review =$this->getAstrologerRatings($user->id,3);

	return apiResponse(true, 'Profile fetched successfully.', $res);
}

private function getAstrologerRatings($astrologerId, $limit = 10)
{
    $ratings= $this->db->table('ratings r')
        ->select('r.id, r.rating, r.review, r.created_at, 
                  u.username, u.profile_image')
        ->join('users u', 'u.id = r.user_id', 'left')
        ->where('r.astrologer_id', $astrologerId)
        ->orderBy('r.created_at', 'DESC')
        ->limit($limit)
        ->get()
        ->getResult();
		foreach ($ratings as &$b) {
        $b->created_at = date('d M Y, h:i A', strtotime($b->created_at));
        $b->profile_image = !empty($b->image)
            ? base_url($b->image)
            : base_url('uploads/defaults/user-default.png');
    }
	return $ratings;
}

	
	public function update_profile()
	{
		$data = sanitize(getRequestData());

		// Validation rules
		$rules = [
			'name'              => 'required',
			'gender'            => 'required|in_list[male,female,other]',
			'specialization_id' => 'required',
			'experience'        => 'required',
			'about'             => 'required',
			'languages'         => 'required',
			'services'          => 'required'
		];

		if (!$this->validateData($data, $rules)) {
			return apiResponse(false, $this->validator->getErrors(), null, 400);
		}

		// Convert stringified arrays if needed
		if (!is_array($data['languages'])) {
			$data['languages'] = json_decode($data['languages'], true);
		}
		if (!is_array($data['services'])) {
			$data['services'] = json_decode($data['services'], true);
		}

		if (!is_array($data['languages']) || !is_array($data['services'])) {
			return apiResponse(false, 'Languages and services must be arrays.', null, 400);
		}

		$user = $this->request->user;

		$submitData = [
			'name'              => $data['name'],
			'gender'            => $data['gender'],
			'specialization_id' => $data['specialization_id'],
			'experience'        => $data['experience'],
			'about'             => $data['about'],
			'address'           => $data['address'],
			'pincode'           => $data['pincode'],
		];

		$this->db->transStart();

		// Update main astrologer profile
		$this->db->table('astrologers')->where('id', $user->id)->update($submitData);

		// Remove previous language/service mappings
		$this->db->table('astrologer_languages')->where('astrologer_id', $user->id)->delete();
		$this->db->table('astrologer_services')->where('astrologer_id', $user->id)->delete();

		// Insert new languages
		foreach ($data['languages'] as $langId) {
			if (is_numeric($langId)) {
				$this->db->table('astrologer_languages')->insert([
					'astrologer_id' => $user->id,
					'language_id'   => $langId
				]);
			}
		}

		// Insert new services
		foreach ($data['services'] as $serviceId) {
			if (is_numeric($serviceId)) {
				$this->db->table('astrologer_services')->insert([
					'astrologer_id' => $user->id,
					'service_id'    => $serviceId
				]);
			}
		}

		$this->db->transComplete();

		if ($this->db->transStatus() === false) {
			return apiResponse(false, 'Failed to update profile.', null, 500);
		}

		return apiResponse(true, 'Profile updated successfully.');
	}

	
	public function update_fcm_token()
	{
		$data = sanitize(getRequestData());

		// Validation rules
		$rules = [
			'fcm_token' => 'required',
		];

		if (!$this->validateData($data, $rules)) {
			return apiResponse(false, $this->validator->getErrors(), null, 400);
		}

		$fcm_token = $data['fcm_token'];

		// Get authenticated user from JWT (middleware should set this)
		$user = $this->request->user;

		// Update FCM token in DB
		$updated = $this->db->table('astrologers')
			->where('id', $user->id)
			->update(['fcm_token' => $fcm_token]);

		if ($updated) {
			return apiResponse(true, 'FCM token updated successfully.');
		}

		return apiResponse(false, 'Failed to update FCM token.', null, 500);
	}

	public function update_profile_image()
	{
		$user = $this->request->user;
		$image = $this->request->getFile('profile_image');

		// Validate input
		if (!$image || !$image->isValid()) {
			return apiResponse(false, 'No image file uploaded or file is invalid.', null, 400);
		}

		// Check file type
		$validTypes = ['jpg', 'jpeg', 'png'];
		if (!in_array($image->getExtension(), $validTypes)) {
			return apiResponse(false, 'Invalid image format. Only JPG, JPEG, PNG allowed.', null, 400);
		}

		// Get old image path
		$oldImage = $this->db->table('astrologers')->select('profile_image')->where('id', $user->id)->get()->getRow('profile_image');

		// Set new file name
		$newName = 'astrologer_' . $user->id . '_' . time() . '.' . $image->getExtension();
		$uploadPath = FCPATH . 'uploads/astrologers/';

		// Create directory if not exists
		if (!is_dir($uploadPath)) {
			mkdir($uploadPath, 0755, true);
		}

		// Move file
		if (!$image->move($uploadPath, $newName)) {
			return apiResponse(false, 'Failed to upload image.', null, 500);
		}

		// Delete old image if exists and not default
		if (!empty($oldImage)) {
			$oldPath = FCPATH . $oldImage;
			if (file_exists($oldPath) && strpos($oldImage, 'default.png') === false) {
				unlink($oldPath);
			}
		}

		// Update user record
		$this->db->table('astrologers')->where('id', $user->id)->update([
			'profile_image' => 'uploads/astrologers/' . $newName
		]);

		return apiResponse(true, 'Profile image updated successfully.', [
			'image_url' => base_url('uploads/astrologers/' . $newName)
		]);
	}
	
	public function profile_setup()
	{
		$data = sanitize(getRequestData());

		// Validation rules
		$rules = [
			'name' => 'required',
			'gender' => 'required|in_list[male,female,other]',
			'specialization_id' => 'required',
			'experience' => 'required',
			'about' => 'required',
			'languages' => 'required',
			'services' => 'required',
			'chat_pricing' => 'required',
			'call_pricing' => 'required',
		];

		if (!$this->validateData($data, $rules)) {
			return apiResponse(false, $this->validator->getErrors(), null, 400);
		}

		// Validate types manually
		if (!is_array($data['languages'])) {
			$data['languages'] = json_decode($data['languages'], true);
		}
		if (!is_array($data['services'])) {
			$data['services'] = json_decode($data['services'], true);
		}

		if (!is_array($data['languages']) || !is_array($data['services'])) {
			return apiResponse(false, 'Languages and services must be arrays.', null, 400);
		}

		$user = $this->request->user;

		$submitData = [
			'name'              => $data['name'],
			'gender'            => $data['gender'],
			'specialization_id' => $data['specialization_id'],
			'experience'        => $data['experience'],
			'about'             => $data['about'],
			'chat_pricing'      => $data['chat_pricing'],
			'call_pricing'      => $data['call_pricing'],
		];

		$this->db->transStart();

		// Update astrologer profile
		$this->db->table('astrologers')->where('id', $user->id)->update($submitData);

		// Clear existing language & service mappings
		$this->db->table('astrologer_languages')->where('astrologer_id', $user->id)->delete();
		$this->db->table('astrologer_services')->where('astrologer_id', $user->id)->delete();

		// Insert languages
		foreach ($data['languages'] as $langId) {
			if (is_numeric($langId)) {
				$this->db->table('astrologer_languages')->insert([
					'astrologer_id' => $user->id,
					'language_id'   => $langId
				]);
			}
		}

		// Insert services
		foreach ($data['services'] as $serviceId) {
			if (is_numeric($serviceId)) {
				$this->db->table('astrologer_services')->insert([
					'astrologer_id' => $user->id,
					'service_id'    => $serviceId
				]);
			}
		}

		$this->db->transComplete();

		if ($this->db->transStatus() === false) {
			return apiResponse(false, 'Failed to setup profile.', null, 500);
		}

		return apiResponse(true, 'Profile setup successfully completed.');
	}

	
	public function list_languages()
	{
		$languages = $this->db->table('languages')
			->select('id, name')
			->orderBy('name', 'ASC')
			->get()
			->getResult();

		return apiResponse(true, 'Languages fetched successfully.', $languages);
	}
	
	public function list_services()
	{
		$services = $this->db->table('services')
			->select('id, name')
			->orderBy('name', 'ASC')
			->get()
			->getResult();

		return apiResponse(true, 'Services fetched successfully.', $services);
	}
	
	public function list_specializations()
	{
		$specializations = $this->db->table('specializations')
			->select('id, name')
			->orderBy('name', 'ASC')
			->get()
			->getResult();

		return apiResponse(true, 'Specializations fetched successfully.', $specializations);
	}
	
	public function save_availability()
	{
		$data = sanitize(getRequestData());
		$user = $this->request->user;

		if (!isset($data['availability']) || !is_array($data['availability'])) {
			return apiResponse(false, 'Availability must be an array.', null, 400);
		}

		$this->db->transStart();

		foreach ($data['availability'] as $entry) {
			if (!isset($entry['day_of_week'], $entry['is_available'])) {
				continue;
			}

			$dayOfWeek = strtolower(trim($entry['day_of_week']));
			$isAvailable = (bool)$entry['is_available'];
			$startTime = $entry['start_time'] ?? null;
			$endTime = $entry['end_time'] ?? null;

			// If available, validate time fields
			if ($isAvailable) {
				if (empty($startTime) || empty($endTime)) {
					return apiResponse(false, "Start and end time required for $dayOfWeek", null, 400);
				}

				if (!preg_match('/^\d{2}:\d{2}$/', $startTime) || !preg_match('/^\d{2}:\d{2}$/', $endTime)) {
					return apiResponse(false, "Invalid time format for $dayOfWeek. Use HH:MM.", null, 400);
				}

				if (strtotime($startTime) >= strtotime($endTime)) {
					return apiResponse(false, "Start time must be before end time for $dayOfWeek", null, 400);
				}
			}

			$updateData = [
				'is_available' => $isAvailable,
				'start_time'   => $startTime,
				'end_time'     => $endTime,
			];

			// Update or insert availability
			$existing = $this->db->table('astrologer_availability')
				->where('astrologer_id', $user->id)
				->where('day_of_week', $dayOfWeek)
				->get()
				->getRow();

			if ($existing) {
				$this->db->table('astrologer_availability')
					->where('id', $existing->id)
					->update($updateData);
			} else {
				$this->db->table('astrologer_availability')->insert(array_merge([
					'astrologer_id' => $user->id,
					'day_of_week'   => $dayOfWeek,
				], $updateData));
			}
		}

		$this->db->transComplete();

		if (!$this->db->transStatus()) {
			return apiResponse(false, 'Failed to update availability.', null, 500);
		}

		return apiResponse(true, 'Availability updated successfully.');
	}

	public function get_availability()
	{
		$user = $this->request->user;

		$rows = $this->db->table('astrologer_availability')
			->where('astrologer_id', $user->id)
			->orderBy('FIELD(day_of_week, "monday", "tuesday", "wednesday", "thursday", "friday", "saturday", "sunday")')
			->get()
			->getResult();

		return apiResponse(true, 'Availability fetched successfully.', $rows);
	}

	public function delete_availability()
{
	$data = sanitize(getRequestData());
	$user = $this->request->user;

	$validDays = ['monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday', 'sunday'];

	if (empty($data['day_of_week'])) {
		return apiResponse(false, 'Day of week is required.', null, 400);
	}

	$day = strtolower(trim($data['day_of_week']));

	if (!in_array($day, $validDays)) {
		return apiResponse(false, 'Invalid day of week. Must be one of: ' . implode(', ', $validDays), null, 400);
	}

	$deleted = $this->db->table('astrologer_availability')
		->where('astrologer_id', $user->id)
		->where('day_of_week', $day)
		->delete();

	if ($this->db->affectedRows() > 0) {
		return apiResponse(true, 'Availability deleted successfully.');
	}

	return apiResponse(false, 'No availability found for this day.', null, 404);
}

public function upcoming_bookings()
{
    $astrologer = $this->request->user; // Assuming astrologer is authenticated
    // $today = date('Y-m-d');

    // Get pagination
    $page  = (int) ($this->request->getVar('page') ?? 1);
    $limit = (int) ($this->request->getVar('limit') ?? 10);
    $offset = ($page - 1) * $limit;

    // Total upcoming bookings count
    $countBuilder = $this->db->table('bookings b')
        ->select('COUNT(DISTINCT b.id) as total')
        ->where('b.astrologer_id', $astrologer->id)
        ->where('b.status', 'confirmed');
        // ->where("DATE(b.booking_date) >=", $today);

    $total = $countBuilder->get()->getRow()->total;

    // Fetch upcoming bookings
    $bookings = $this->db->table('bookings b')
        ->select("
            b.id,
            b.booking_id,
            b.user_id,
            b.booking_date,
            b.time_slot,
            b.slot_duration,
            b.amount,
            b.status,
            u.name as user_name,
            u.profile_image as user_image,
            u.gender,
            u.dob,
			s.name as service_name,
			a.call_pricing,
			a.chat_pricing,
        ")
        ->join('users u', 'u.id = b.user_id', 'left')
        ->join('astrologers a', 'a.id = b.astrologer_id')
		->join('services s', 's.id = b.service_id', 'left')
        ->where('b.astrologer_id', $astrologer->id)
        ->where('b.status', 'confirmed')
        // ->where("DATE(b.booking_date) >=", $today)
        ->orderBy('b.booking_date', 'ASC')
        ->orderBy('b.id', 'ASC')
        ->limit($limit, $offset)
        ->get()->getResult();

    // Format response
    foreach ($bookings as &$b) {
        $b->booking_date = date('d M Y, h:i A', strtotime($b->booking_date . ' ' . explode('-', $b->time_slot)[0]));
        $b->user_image = !empty($b->user_image)
            ? base_url($b->user_image)
            : base_url('uploads/defaults/user-default.png');
    }

    $response = [
        'current_page'      => (string)$page,
        'total_pages'       => (string)ceil($total / $limit),
        'total_records'     => (string) $total,
        'records_per_page'  => (string)$limit,
        'data'              => $bookings
    ];

    return apiResponse(true, 'Upcoming bookings fetched successfully', $response);
}

private function getUpcomingBookings($astrologerId, $limit = 10, $offset = 0)
{
    $bookings= $this->db->table('bookings b')
        ->select("b.id, b.booking_id, b.user_id, b.booking_date, b.time_slot, b.slot_duration, b.amount, b.status,
                  u.name as user_name, u.profile_image as user_image, u.gender, u.dob,
                  s.name as service_name, a.call_pricing, a.chat_pricing")
        ->join('users u', 'u.id = b.user_id', 'left')
        ->join('astrologers a', 'a.id = b.astrologer_id')
        ->join('services s', 's.id = b.service_id', 'left')
        ->where('b.astrologer_id', $astrologerId)
        ->where('b.status', 'confirmed')
        ->orderBy('b.booking_date', 'ASC')
        ->orderBy('b.id', 'ASC')
        ->limit($limit, $offset)
        ->get()
        ->getResult();
		foreach ($bookings as &$b) {
			$b->booking_date = date('d M Y, h:i A', strtotime($b->booking_date . ' ' . explode('-', $b->time_slot)[0]));
			$b->user_image = !empty($b->user_image) ? base_url($b->user_image) : base_url('uploads/defaults/user-default.png');
		}
	return $bookings;
}
public function new_bookings_request()
{
    $astrologer = $this->request->user; // Assuming astrologer is authenticated
    // $today = date('Y-m-d');

    // Get pagination
    $page  = (int) ($this->request->getVar('page') ?? 1);
    $limit = (int) ($this->request->getVar('limit') ?? 10);
    $offset = ($page - 1) * $limit;

    // Total upcoming bookings count
    $countBuilder = $this->db->table('bookings b')
        ->select('COUNT(DISTINCT b.id) as total')
        ->where('b.astrologer_id', $astrologer->id)
        ->where('b.status', 'confirmed');
        // ->where("DATE(b.booking_date) >=", $today);

    $total = $countBuilder->get()->getRow()->total;

    // Fetch upcoming bookings
    $bookings = $this->db->table('bookings b')
        ->select("
            b.id,
            b.booking_id,
            b.user_id,
            b.booking_date,
            b.time_slot,
            b.slot_duration,
            b.amount,
            b.status,
            u.name as user_name,
            u.profile_image as user_image,
            u.gender,
            u.dob,
			s.name as service_name,
			a.call_pricing,
			a.chat_pricing,
        ")
        ->join('users u', 'u.id = b.user_id', 'left')
        ->join('astrologers a', 'a.id = b.astrologer_id')
		->join('services s', 's.id = b.service_id', 'left')
        ->where('b.astrologer_id', $astrologer->id)
        ->where('b.status', 'pending')
        // ->where("DATE(b.booking_date) >=", $today)
        ->orderBy('b.booking_date', 'ASC')
        ->orderBy('b.id', 'ASC')
        ->limit($limit, $offset)
        ->get()->getResult();

    // Format response
    foreach ($bookings as &$b) {
        $b->booking_date = date('d M Y, h:i A', strtotime($b->booking_date . ' ' . explode('-', $b->time_slot)[0]));
        $b->user_image = !empty($b->user_image)
            ? base_url($b->user_image)
            : base_url('uploads/defaults/user-default.png');
    }

    $response = [
        'current_page'      => (string)$page,
        'total_pages'       => (string)ceil($total / $limit),
        'total_records'     => (string) $total,
        'records_per_page'  => (string)$limit,
        'data'              => $bookings
    ];

    return apiResponse(true, 'New bookings request fetched successfully', $response);
}

public function booking_history()
{
    $astrologer = $this->request->user;

    // Get filters
    $page       = (int) ($this->request->getVar('page') ?? 1);
    $limit      = (int) ($this->request->getVar('limit') ?? 10);
    $offset     = ($page - 1) * $limit;
    $search     = trim($this->request->getVar('search'));
    $service_id = $this->request->getVar('service_id');
    $from_date  = $this->request->getVar('from_date'); // format: YYYY-MM-DD
    $to_date    = $this->request->getVar('to_date');   // format: YYYY-MM-DD

    // Build the base query
    $builder = $this->db->table('bookings b')
        ->join('users u', 'u.id = b.user_id', 'left')
        ->join('astrologers a', 'a.id = b.astrologer_id')
        ->join('services s', 's.id = b.service_id', 'left')
        ->where('b.astrologer_id', $astrologer->id)
        ->whereIn('b.status', ['completed', 'cancelled']);

    // Apply search filter
    if (!empty($search)) {
        $builder->groupStart()
            ->like('u.name', $search)
            ->orLike('b.booking_id', $search)
            ->groupEnd();
    }

    // Apply service filter
    if (!empty($service_id)) {
        $builder->where('b.service_id', $service_id);
    }

    // Apply date range filter
    if (!empty($from_date) && !empty($to_date)) {
        $builder->where("DATE(b.booking_date) >=", $from_date);
        $builder->where("DATE(b.booking_date) <=", $to_date);
    }

    // Clone for total count
    $countBuilder = clone $builder;
    $total = $countBuilder->select('COUNT(DISTINCT b.id) as total')->get()->getRow()->total;

    // Get paginated records
    $bookings = $builder
        ->select("
            b.id,
            b.booking_id,
            b.user_id,
            b.booking_date,
            b.time_slot,
            b.slot_duration,
            b.amount,
            b.status,
            b.cancelled_by,
            u.name as user_name,
            u.profile_image as user_image,
            u.gender,
            u.dob,
            s.name as service_name,
            a.call_pricing,
            a.chat_pricing
        ")
        ->orderBy('b.booking_date', 'ASC')
        ->orderBy('b.id', 'ASC')
        ->limit($limit, $offset)
        ->get()
        ->getResult();

    // Format output
    foreach ($bookings as &$b) {
        $b->booking_date = date('d M Y, h:i A', strtotime($b->booking_date . ' ' . explode('-', $b->time_slot)[0]));
        $b->user_image = !empty($b->user_image)
            ? base_url($b->user_image)
            : base_url('uploads/defaults/user-default.png');
    }

    return apiResponse(true, 'Booking history fetched successfully', [
        'current_page'      => (string) $page,
        'total_pages'       => (string) ceil($total / $limit),
        'total_records'     => (string) $total,
        'records_per_page'  => (string) $limit,
        'data'              => $bookings
    ]);
}

private function getRecentBooking($astrologerId,$limit=3){
	 $builder = $this->db->table('bookings b')
        ->join('users u', 'u.id = b.user_id', 'left')
        ->join('astrologers a', 'a.id = b.astrologer_id')
        ->join('services s', 's.id = b.service_id', 'left')
        ->where('b.astrologer_id', $astrologerId)
        ->where('b.status', 'completed');

	$bookings = $builder
        ->select("
            b.id,
            b.booking_id,
            b.user_id,
            b.booking_date,
            b.time_slot,
            b.slot_duration,
            b.amount,
            b.status,
            b.cancelled_by,
            u.name as user_name,
            u.profile_image as user_image,
            u.gender,
            u.dob,
            s.name as service_name,
            a.call_pricing,
            a.chat_pricing
        ")
        ->orderBy('b.booking_date', 'ASC')
        ->orderBy('b.id', 'ASC')
        ->limit($limit)
        ->get()
        ->getResult();

    // Format output
    foreach ($bookings as &$b) {
        $b->booking_date = date('d M Y, h:i A', strtotime($b->booking_date . ' ' . explode('-', $b->time_slot)[0]));
        $b->user_image = !empty($b->user_image)
            ? base_url($b->user_image)
            : base_url('uploads/defaults/user-default.png');
    }
	return $bookings;
}
public function booking_status()
{
    $data = sanitize(getRequestData());
    $user = $this->request->user;

    $rules = [
        'booking_id' => 'required',
        'status'     => 'required|in_list[confirmed,cancelled]',
    ];

    if (!$this->validateData($data, $rules)) {
        return apiResponse(false, $this->validator->getErrors(), null, 400);
    }

    $booking = $this->db->table('bookings')->where([
        'id'            => $data['booking_id'],
        'astrologer_id' => $user->id,
    ])->get()->getRow();

    if (!$booking) {
        return apiResponse(false, 'Booking not found.', null, 404);
    }

    // Prevent status change if already set
    if ($booking->status === 'cancelled' && $data['status'] === 'confirmed') {
        return apiResponse(false, 'Cannot confirm a cancelled booking.', null, 400);
    }

    if ($booking->status === 'confirmed' && $data['status'] === 'cancelled') {
        return apiResponse(false, 'Cannot cancel a confirmed booking.', null, 400);
    }
	if ($booking->status === $data['status']) {
		return apiResponse(false, 'Booking already marked as ' . $data['status'] . '.');
	}

    $this->db->transStart();

    // If status is "cancelled"
    if ($data['status'] === 'cancelled') {
        $this->db->table('bookings')->where('id', $booking->id)->update([
            'status'        => 'cancelled',
            'cancelled_by'  => 'astrologer',
            'updated_at'    => $this->data['datetime']
        ]);

        // Refund if not already refunded
        if ((int)$booking->refunded === 0) {
            $userRow         = $this->db->table('users')->where('id', $booking->user_id)->get()->getRow();
            $opening_balance = (float)$userRow->wallet_amount;
            $refund_amount   = (float)$booking->amount;
            $closing_balance = $opening_balance + $refund_amount;

            // Wallet transaction
            $this->db->table('wallet_transactions')->insert([
                'user_id'          => $booking->user_id,
                'payment_id'       => $booking->booking_id,
                'transaction_type' => 'credit',
                'amount'           => $refund_amount,
                'opening_balance'  => $opening_balance,
                'closing_balance'  => $closing_balance,
                'description'      => 'Refund for cancelled booking ID ' . $booking->booking_id,
                'created_at'       => $this->data['datetime']
            ]);

            // Update wallet
            $this->db->table('users')->where('id', $booking->user_id)->update([
                'wallet_amount' => $closing_balance
            ]);

            // Set refunded flag
            $this->db->table('bookings')->where('id', $booking->id)->update([
                'refunded' => 1
            ]);
        }

        $message = 'Booking cancelled and amount refunded successfully.';
    }

    // If status is "confirmed"
    if ($data['status'] === 'confirmed') {
        $this->db->table('bookings')->where('id', $booking->id)->update([
            'status'     => 'confirmed',
            'updated_at' => $this->data['datetime']
        ]);
        $message = 'Booking confirmed successfully.';
    }

    $this->db->transComplete();

    if (!$this->db->transStatus()) {
        return apiResponse(false, 'Status update failed due to internal error.', null, 500);
    }

    return apiResponse(true, $message);
}


	public function logout()
	{
		$user = $this->request->user;

		// Delete all access/refresh tokens
		$this->db->table('user_tokens')
			->where('user_id', $user->id)
			->where('user_type', $user->user_type)
			->delete();

		// Blank out FCM token in users table
		$this->db->table('astrologers')
			->where('id', $user->id)
			->update(['fcm_token' => null]);

		return apiResponse(true, 'Logged out successfully.');
	}

	public function delete_account()
	{
		$user = $this->request->user;

		// Delete all access/refresh tokens
		$this->db->table('user_tokens')
			->where('user_id', $user->id)
			->where('user_type', $user->user_type)
			->delete();

		// Blank out FCM token in users table
		$this->db->table('astrologers')
			->where('id', $user->id)
			->update(['fcm_token' => null,'delete_status'=>'true']);

		return apiResponse(true, 'Account deleted successfully.');
	}

    
}
